<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the dashboard.
 *
 * @link       http://example.com
 * @since      1.0.0
 *
 * @package    DT_Dummy
 * @subpackage DT_Dummy/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, dashboard-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    DT_Dummy
 * @subpackage DT_Dummy/includes
 * @author     Dream-Theme
 */
class DT_Dummy {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      DT_Dummy_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Plugin dir.
	 */
	protected $plugin_dir;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the Dashboard and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		$this->plugin_name = 'dt-dummy';
		$this->version = '1.5.3';
		$this->plugin_dir = trailingslashit( plugin_dir_path( dirname( __FILE__ ) ) );

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - DT_Dummy_Loader. Orchestrates the hooks of the plugin.
	 * - DT_Dummy_i18n. Defines internationalization functionality.
	 * - DT_Dummy_Admin. Defines all hooks for the dashboard.
	 * - DT_Dummy_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {
		$plugin_dir = $this->get_plugin_dir();

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once $plugin_dir . 'includes/class-dt-dummy-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once $plugin_dir . 'includes/class-dt-dummy-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the Dashboard.
		 */
		require_once $plugin_dir . 'admin/class-dt-dummy-admin.php';

		require_once $plugin_dir . 'admin/dt-dummy-admin-functions.php';
		require_once $plugin_dir . 'admin/class-dt-dummy-admin-phpstatus.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once $plugin_dir . 'public/class-dt-dummy-public.php';

		require_once $plugin_dir . 'includes/class-dt-dummy-content.php';

		$this->loader = new DT_Dummy_Loader();

		/**
		 * Load admin utility classes.
		 */
		require $plugin_dir . 'admin/interface-dt-dummy-plugins-checker.php';
		require $plugin_dir . 'admin/class-dt-dummy-tgmpa.php';
		require $plugin_dir . 'admin/class-dt-dummy-wp-plugins.php';
		require $plugin_dir . 'includes/class-dt-dummy-import-manager.php';

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the DT_Dummy_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new DT_Dummy_i18n();
		$plugin_i18n->set_domain( $this->get_plugin_name() );

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the dashboard functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new DT_Dummy_Admin( $this );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// setup dummy
		$this->loader->add_action( 'after_setup_theme', $plugin_admin, 'choose_dummy_content' );

		// add admin pages
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_page' );

		// ajax handler
		$this->loader->add_action( 'wp_ajax_presscore_import_dummy', $plugin_admin, 'ajax_response' );
		$this->loader->add_action( 'wp_ajax_dt_dummy_php_status', $plugin_admin, 'get_php_ini_status' );

		// allow export attachments
		$this->loader->add_action( 'export_filters', $plugin_admin, 'allow_export_additional_post_types' );

		// add admin notice
		$this->loader->add_action( 'admin_notices', $plugin_admin, 'add_admin_notices' );

		// add plugin action links
		$this->loader->add_action( 'plugin_action_links_' . plugin_basename( DT_DUMMY_PLUGIN_MAIN_FILE ), $plugin_admin, 'add_plugin_action_links', 10, 2 );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    DT_Dummy_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Retrieve plugin dir.
	 * 
	 * @return string
	 */
	public function get_plugin_dir( $path = '' ) {
		return $this->plugin_dir . $path;
	}

}
